#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2005, 2006, 2010, 2011 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching	<crunchy@bitkipper.net>

"""
a generic signal implementation for propagating asynchron events
"""

import sys
from typing import Dict  # noqa: F401

if sys.version_info >= (3,):
	basestring = (str)

__signals = {}  # type: Dict

# exception classes


class UnknownSignalError(Exception):
	pass


class SignalExistsError(Exception):
	pass


class Signal(object):

	def __init__(self, name):
		self.name = name
		self.__callbacks = []

	def emit(self, *args):
		for cb in self.__callbacks[:]:
			if cb(*args):
				self.disconnect(cb)

	def connect(self, callback):
		self.__callbacks.append(callback)

	def disconnect(self, callback):
		try:
			self.__callbacks.remove(callback)
		except ValueError:
			pass

	def __str__(self):
		return self.name


class Provider(object):

	def __init__(self):
		self.__signals = {}

	def signal_new(self, signal):
		new(signal, self.__signals)

	def signal_exists(self, signal):
		return exists(signal, self.__signals)

	def signal_connect(self, signal, callback):
		connect(signal, callback, self.__signals)

	def signal_disconnect(self, signal, callback):
		disconnect(signal, callback, self.__signals)

	def signal_emit(self, signal, *args):
		if isinstance(signal, Signal) and signal.name in self.__signals:
			self.__signals[signal.name].emit(*args)
		elif isinstance(signal, basestring) and signal in self.__signals:
			self.__signals[signal].emit(*args)


def _select_signals(signals):
	if signals is None:
		return __signals
	return signals


def new(signal, signals=None):
	_signals = _select_signals(signals)
	if isinstance(signal, basestring):
		signal = Signal(signal)

	if signal.name in _signals:
		raise SignalExistsError("Signal '%s' already exists" % signal.name)
	else:
		_signals[signal.name] = signal


def exists(signal, signals=None):
	_signals = _select_signals(signals)
	if isinstance(signal, basestring):
		return signal in _signals
	else:
		return signal.name in _signals


def connect(signal, callback, signals=None):
	_signals = _select_signals(signals)
	if isinstance(signal, Signal) and signal.name in _signals:
		_signals[signal.name].connect(callback)
	elif isinstance(signal, basestring):
		if signal in _signals:
			_signals[signal].connect(callback)
		else:
			raise UnknownSignalError("unknown signal '%s'" % signal)


def disconnect(signal, callback, signals=None):
	_signals = _select_signals(signals)
	if isinstance(signal, Signal) and signal.name in _signals:
		_signals[signal.name].disconnect(callback)
	elif isinstance(signal, basestring) and signal in _signals:
		_signals[signal].disconnect(callback)


def emit(signal, *args):
	if isinstance(signal, Signal) and signal.name in __signals:
		__signals[signal.name].emit(*args)
	elif isinstance(signal, basestring):
		if signal in __signals:
			__signals[signal].emit(*args)
		else:
			raise UnknownSignalError("unknown signal '%s'" % signal)
