#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
child process control using notifier.
"""

import fcntl
import os
import shlex
import subprocess
import sys
import tempfile
import time
from typing import IO, Callable, List, Optional, Union, cast  # noqa: F401

import notifier

from . import _FileLike, _TimerID, log, signals  # noqa: F401

__all__ = ['Process', 'RunIt', 'Shell']


if sys.version_info >= (3,):
	basestring = str


_processes = []  # type: List[Process]


class Process(signals.Provider):
	"""
	Base class for starting child processes and monitoring standard
	output and error.
	"""

	def __init__(self, cmd, stdout=True, stderr=True, shell=False):
		# type: (Union[List[str], str], bool, bool, bool) -> None
		""" Init the child process 'cmd'. This can either be a string or a list
		of arguments (similar to popen2). stdout and stderr of the child
		process can be handled by connecting to the signals 'stdout'
		and/or 'stderr'. The signal functions have one argument that is
		of type list and contains all new lines. By setting one of the
		boolean arguments 'stdout' and 'stderr' to False the monitoring
		of these files can be deactivated.


		Signals:
		'stderr' ( pid, line )
				emitted when the IO_Handler reported another line of input
				(from stdout).
				pid			: PID of the process that produced the output
				line		: line of output
		'stdout' ( pid, line )
				emitted when the IO_Handler reported another line of input
				(from stderr).
				pid			: PID of the process that produced the output
				line		: line of output
		"""
		signals.Provider.__init__(self)
		if stderr:
			self.signal_new('stderr')
		if stdout:
			self.signal_new('stdout')
		self.signal_new('killed')

		if not shell and not isinstance(cmd, (list, tuple)):
			self._cmd = shlex.split(str(cmd))  # type: Union[List[str], str] # shlex.split can not handle Unicode strings
		else:
			self._cmd = cmd

		self._shell = shell
		if not shell and self._cmd:
			self._name = self._cmd[0].split('/')[-1]
		else:
			self._name = '<unknown>'
		self.stopping = False
		self.pid = None  # type: Optional[int]
		self.child = None  # type: Optional[subprocess.Popen]

		self.__dead = True
		self.__kill_timer = None  # type: Optional[_TimerID]

		if not _processes:
			notifier.dispatcher_add(_watcher)
		_processes.append(self)

	def _read_stdout(self, line):
		# type: (str) -> None
		"""emit signal 'stdout', announcing that there is another line
		of output"""
		self.signal_emit('stdout', self.pid, line)

	def _read_stderr(self, line):
		# type: (str) -> None
		"""emit signal 'stdout', announcing that there is another line
		of output"""
		self.signal_emit('stderr', self.pid, line)

	def start(self, args=None):
		# type: (Optional[str]) -> int
		"""
		Starts the process.	 If args is not None, it can be either a list or
		string, as with the constructor, and is appended to the command line
		specified in the constructor.
		"""
		if not self.__dead:
			raise SystemError("process is already running.")
		if self.stopping:
			raise SystemError("process is currently dying.")

		if not args:
			cmd = self._cmd  # type: Union[List[str], str]
		elif self._shell:
			cmd = '%s %s' % (self._cmd, args)
		else:
			cmd = cast(List, self._cmd) + shlex.split(args)

		self.__kill_timer = None
		self.__dead = False
		self.binary = cmd[0]

		self.stdout = self.stderr = None  # type: Optional[IO_Handler]
		if not self.signal_exists('stdout') and not self.signal_exists('stderr'):
			self.pid = subprocess.Popen(cmd, shell=self._shell).pid
		else:
			stdout = subprocess.PIPE if self.signal_exists('stdout') else None
			stderr = subprocess.PIPE if self.signal_exists('stderr') else None

			# line buffered, no shell
			self.child = subprocess.Popen(cmd, bufsize=1, shell=self._shell, stdout=stdout, stderr=stderr)
			self.pid = self.child.pid

			if self.child.stdout:
				# IO_Handler for stdout
				self.stdout = IO_Handler('stdout', self.child.stdout, self._read_stdout, self._name)
				self.stdout.signal_connect('closed', self._closed)

			if self.child.stderr:
				# IO_Handler for stderr
				self.stderr = IO_Handler('stderr', self.child.stderr, self._read_stderr, self._name)
				self.stderr.signal_connect('closed', self._closed)

		log.info('running %s (pid=%s)' % (self.binary, self.pid))

		return self.pid

	def dead(self, pid, status):
		# type: (int, int) -> None
		self.__dead = True
		# check io handlers if there is pending output
		for output in (self.stdout, self.stderr):
			if output:
				output._handle_input(flush_partial_lines=True)
		self.signal_emit('killed', pid, status)

	def _closed(self, name):
		# type: (str) -> None
		if name == 'stderr':
			self.stderr = None
		elif name == 'stdout':
			self.stdout = None

		if not self.stdout and not self.stderr:
			try:
				assert self.pid
				pid, status = os.waitpid(self.pid, os.WNOHANG)
				if pid:
					self.dead(pid, status)
			except OSError:  # already dead and buried
				pass

	def write(self, line):
		# type: (str) -> None
		"""
		Pass a string to the process
		"""
		try:
			assert self.child
			self.child.communicate(line)
		except (IOError, ValueError):
			pass

	def is_alive(self):
		# type: () -> bool
		"""
		Return True if the process is still running
		"""
		return not self.__dead

	def stop(self):
		# type: () -> None
		"""
		Stop the child. Tries to kill the process with signal 15 and after that
		kill -9 will be used to kill the app.
		"""
		if self.stopping:
			return

		self.stopping = True

		if self.is_alive() and not self.__kill_timer:
			cb = notifier.Callback(self.__kill, 15)
			self.__kill_timer = notifier.timer_add(0, cb)

	def __kill(self, signal):
		# type: (int) -> bool
		"""
		Internal kill helper function
		"""
		if not self.is_alive():
			self.__dead = True
			self.stopping = False
			return False
		# child needs some assistance with dying ...
		try:
			assert self.pid is not None
			os.kill(self.pid, signal)
		except OSError:
			pass

		if signal == 15:
			cb = notifier.Callback(self.__kill, 9)
			self.__kill_timer = notifier.timer_add(3000, cb)

		return False


def _watcher():
	# type: () -> bool
	finished = []

	for proc in _processes:
		try:
			if not proc.pid:
				continue
			pid, status = os.waitpid(proc.pid, os.WNOHANG)
			if pid:
				proc.dead(pid, status)
				finished.append(proc)
		except OSError:  # already dead and buried
			finished.append(proc)

	for i in finished:
		_processes.remove(i)

	return bool(_processes)


class IO_Handler(signals.Provider):
	"""
	Reading data from socket (stdout or stderr)

	Signals:
	'closed' ( name )
			emitted when the file was closed.
			name			: name of the IO_Handler
	"""

	def __init__(self, name, fp, callback, logger=None):
		# type: (str, IO[bytes], Callable, Optional[str]) -> None
		signals.Provider.__init__(self)

		self.name = name
		self.fp = fp
		fcntl.fcntl(self.fp.fileno(), fcntl.F_SETFL, os.O_NONBLOCK)
		self.callback = callback
		self.saved = b''
		notifier.socket_add(fp, self._handle_input)
		self.signal_new('closed')

	def close(self):
		# type: () -> None
		"""
		Close the IO to the child.
		"""
		notifier.socket_remove(self.fp)
		self.fp.close()
		self.signal_emit('closed', self.name)

	def _handle_input(self, socket=None, flush_partial_lines=False):
		# type: (_FileLike, bool) -> bool
		"""
		Handle data input from socket.
		"""
		try:
			self.fp.flush()
			data = self.fp.read(65535)
		except IOError as exp:
			if exp.errno == 11:
				# Resource temporarily unavailable; if we try to read on a
				# non-blocking descriptor we'll get this message.
				return True
			data = None

		if not data:
			if self.saved:
				# Although socket has no data anymore, we still have data left
				# over in the buffer.
				self.flush_buffer()
				return True
			self.close()
			return False

		data = data.replace(b'\r', b'\n')
		partial_line = data[-1] != b'\n'
		lines = data.split(b'\n')

		# split creates an empty line of string ends with line break
		if not lines[-1]:
			del lines[-1]
		# prepend saved data to first line
		if self.saved:
			lines[0] = self.saved + lines[0]
			self.saved = b''
		# Only one partial line?
		if partial_line and not flush_partial_lines:
			self.saved = lines[-1]
			del lines[-1]

		# send lines
		self.callback(lines)

		return True

	def flush_buffer(self):
		# type: () -> None
		if self.saved:
			self.callback(self.saved.split(b'\n'))
			self.saved = b''


class RunIt(Process):
	"""Is a more simple child process handler based on Process that
	caches the output and provides it to the caller with the signal
	'finished'.

	Signals:
	'finished' ( pid, status[, stdout[, stderr ] ] )
			emitted when the child process is dead.
			pid				: process ID
			status			: exit code of the child process
			stdout, stderr	: are only provided when stdout and/or stderr is monitored
			"""

	def __init__(self, command, stdout=True, stderr=False, shell=False):
		# type: (Union[List[str], str], bool, bool, bool) -> None
		Process.__init__(self, command, stdout=stdout, stderr=stderr, shell=shell)

		if stdout:
			self.__stdout = []  # type: Optional[List[str]]
			cb = notifier.Callback(self._output, self.__stdout)
			self.signal_connect('stdout', cb)
		else:
			self.__stdout = None

		if stderr:
			self.__stderr = []  # type: Optional[List[str]]
			cb = notifier.Callback(self._output, self.__stderr)
			self.signal_connect('stderr', cb)
		else:
			self.__stderr = None

		self.signal_connect('killed', self._finished)
		self.signal_new('finished')

	def _output(self, pid, line, buffer):
		# type: (int, Union[List[str], str], List[str]) -> None
		if isinstance(line, list):
			buffer.extend(line)
		else:
			buffer.append(line)

	def _finished(self, pid, status):
		# type: (int, int) -> None
		exit_code = os.WEXITSTATUS(status)
		if self.__stdout is not None:
			if self.__stderr is None:
				self.signal_emit('finished', pid, exit_code, self.__stdout)
			else:
				self.signal_emit('finished', pid, exit_code, self.__stdout, self.__stderr)
		elif self.__stderr is not None:
			self.signal_emit('finished', pid, exit_code, self.__stderr)
		else:
			self.signal_emit('finished', pid, exit_code)


class Shell(RunIt):
	'''A simple interface for running shell commands as child processes'''

	def __init__(self, command, stdout=True, stderr=False):
		# type: (Union[List[str], str], bool, bool) -> None
		RunIt.__init__(self, command, stdout=stdout, stderr=stderr, shell=True)


class CountDown(object):
	'''This class provides a simple method to measure the expiration of
	an amount of time'''

	def __init__(self, timeout):
		# type: (float) -> None
		self.start = time.time() * 1000
		self.timeout = timeout

	def __call__(self):
		# type: () -> bool
		now = time.time() * 1000
		return not self.timeout or (now - self.start < self.timeout)


class Child(object):
	'''Describes a child process and is used for return values of the
	run method.'''

	def __init__(self, stdout=None, stderr=None):
		# type: (Optional[IO[bytes]], Optional[IO[bytes]]) -> None
		self.pid = None  # type: Optional[int]
		self.exitcode = None  # type: Optional[int]
		self.stdout = stdout
		self.stderr = stderr


def run(command, timeout=0, stdout=True, stderr=True, shell=True):
	# type: (Union[List[str], str], float, bool, bool, bool) -> Child
	'''Runs a child process with the <command> and waits <timeout>
	seconds for its termination. If <stdout> is True the standard output
	is written to a temporary file. The same can be done for the standard
	error output with the argument <stderr>. If <shell> is True the
	command is passed to a shell. The return value is a Child
	object. The member variable <pid> is set if the process is still
	running after <timeout> seconds otherwise <exitcode> is set.'''
	# a dispatcher function required to activate the minimal timeout
	def fake_dispatcher():
		return True
	notifier.dispatcher_add(fake_dispatcher)

	countdown = CountDown(timeout)
	out = err = None
	if stdout:
		out = tempfile.NamedTemporaryFile()
	if stderr:
		err = tempfile.NamedTemporaryFile()

	if isinstance(command, basestring):
		command = shlex.split(command)
	child = subprocess.Popen(command, shell=shell, stdout=out, stderr=err)

	while countdown():
		exitcode = child.poll()
		if exitcode is not None:
			break
		notifier.step()

	# remove dispatcher function
	notifier.dispatcher_remove(fake_dispatcher)

	# prepare return code
	ret = Child(stdout=out, stderr=err)
	if child.returncode is None:
		ret.pid = child.pid
	else:
		# move to beginning of files
		if out:
			out.seek(0)
		if err:
			err.seek(0)
		ret.exitcode = child.returncode

	return ret


def kill(pid, signal=15, timeout=0):
	# type: (int, int, float) -> Optional[int]
	'''kills the process specified by pid that may be a process id or a
	Child object. The process is killed with the provided signal (by
	default 15). If the process is not dead after <timeout> seconds the
	function exist anyway'''
	# a dispatcher function required to activate the minimal timeout
	def fake_dispatcher():
		return True
	notifier.dispatcher_add(fake_dispatcher)

	if isinstance(pid, Child):
		if pid.pid:
			pid = pid.pid
		else:
			return pid.exitcode

	os.kill(pid, signal)
	countdown = CountDown(timeout)
	while countdown():
		dead_pid, sts = os.waitpid(pid, os.WNOHANG)
		if dead_pid == pid:
			break
		notifier.step()
	else:
		# remove dispatcher function
		notifier.dispatcher_remove(fake_dispatcher)
		return None

	# remove dispatcher function
	notifier.dispatcher_remove(fake_dispatcher)

	if os.WIFSIGNALED(sts):
		return -os.WTERMSIG(sts)

	return os.WEXITSTATUS(sts)
