#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2004, 2005, 2006, 2007 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
Simple mainloop that watches sockets and timers.
notifier wrapper for GTK+ 2.x
"""

from __future__ import absolute_import

from typing import Any, Callable, Dict, Optional  # noqa: F401

import gobject

from . import _DispathCB, _FileLike, _ReShedule, _SocketCB, _TimerCB, log  # noqa: F401

IO_READ = gobject.IO_IN
IO_WRITE = gobject.IO_OUT
IO_EXCEPT = gobject.IO_ERR
_TimerID = int

_options = {
	'x11': True,
}

# map of Sockets/Methods -> GTK source IDs
_gtk_socketIDs = {
	IO_READ: {},
	IO_WRITE: {},
}  # type: Dict[int, Dict[_FileLike, int]]
_gtk_idleIDs = {}  # type: Dict[_DispathCB, int]


def socket_add(socket, method, condition=IO_READ):
	# type: (_FileLike, _SocketCB, int) -> None
	"""The first argument specifies a socket, the second argument has to be a
	function that is called whenever there is data ready in the socket."""
	source = gobject.io_add_watch(socket, condition, _socket_callback, method)
	_gtk_socketIDs[condition][socket] = source


def _socket_callback(source, condition, method):
	# type: (_FileLike, int, _SocketCB) -> _ReShedule
	"""This is an internal callback function, that maps the GTK source IDs
	to the socket objects that are used by pynotifier as an identifier
	"""
	if source in _gtk_socketIDs[condition]:
		ret = method(source)
		if not ret:
			socket_remove(source, condition)
		return ret

	log.info("socket '%s' not found" % source)
	return False


def socket_remove(socket, condition=IO_READ):
	# type: (_FileLike, int) -> None
	"""Removes the given socket from scheduler."""
	if socket in _gtk_socketIDs[condition]:
		gobject.source_remove(_gtk_socketIDs[condition][socket])
		del _gtk_socketIDs[condition][socket]
	else:
		log.info("socket '%s' not found" % socket)


def timer_add(interval, method):
	# type: (int, _TimerCB) -> _TimerID
	"""The first argument specifies an interval in milliseconds, the
	second argument a function. This is function is called after
	interval seconds. If it returns true it's called again after
	interval seconds, otherwise it is removed from the scheduler. The
	third (optional) argument is a parameter given to the called
	function."""
	return gobject.timeout_add(interval, method)


def timer_remove(id):
	# type: (_TimerID) -> None
	"""Removes the timer specified by id from the scheduler."""
	gobject.source_remove(id)


def dispatcher_add(func):
	# type: (_DispathCB) -> None
	if func not in _gtk_idleIDs:
		_gtk_idleIDs[func] = gobject.idle_add(func)


def dispatcher_remove(func):
	# type: (_DispathCB) -> None
	if func in _gtk_idleIDs:
		gobject.source_remove(_gtk_idleIDs[func])


_mainloop = None  # type: Optional[Any]
_step = None  # type: Optional[Callable[[bool], None]]


def step(sleep=True, external=True):
	# type: (bool, bool) -> None
	assert _step
	_step(sleep)


def loop():
	# type: () -> None
	"""Execute main loop forever."""
	while True:
		step()


def _init():
	# type: () -> None
	global _step, _mainloop

	if _options['x11']:
		import gtk
		_step = gtk.main_iteration_do
	else:
		_mainloop = gobject.main_context_default()
		_step = _mainloop.iteration

	gobject.threads_init()
