#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: LGPL-2.1-only
# Copyright (C) 2004, 2005, 2006 Andreas Büsching <crunchy@bitkipper.net>
# Copyright 2015-2022 Univention GmbH
# Author: Andreas Büsching <crunchy@bitkipper.net>

"""
Simple mainloop that watches sockets and timers.
"""

from __future__ import absolute_import

from typing import Any, Callable, Generic, NewType, Optional, TypeVar, Union  # noqa: F401

try:
	from typing import Protocol
except ImportError:
	try:
		from typing_extension import Protocol  # type: ignore
	except ImportError:
		Protocol = object  # type: ignore

from . import log  # noqa: F401
from .version import VERSION  # noqa: F401

_T = TypeVar("_T")


def _get_fd(obj):
	# type: (_FileLike) -> int
	"""Returns a file descriptor. obj can be a file descriptor or an
	object of type socket.socket, file or socket._socketobject"""
	if isinstance(obj, int):
		return obj
	if hasattr(obj, 'fileno'):
		return obj.fileno()
	raise TypeError("argument must be an int, or have a fileno() method.")


class _HasFileno(Protocol):
	"""
	A :py:class:`file` like object with a :py:meth:`fileno` method.
	"""
	def fileno(self):
		# type: () -> int
		pass


_FileLike = Union[_HasFileno, int]
_ReShedule = Optional[bool]


class __SocketCB(Protocol):
	"""
	Signature for a callback function called on socket activity.
	"""
	def __call__(self, __sock):
		# type: (_FileLike) -> _ReShedule
		pass


_SocketCB = Union[__SocketCB, "Callback"]


class __TimerCB(Protocol):
	"""
	Signature for a callback function called when time expires.
	"""
	def __call__(self):
		# type: () -> _ReShedule
		pass


_TimerCB = Union[__TimerCB, "Callback"]


_TimerID = Any
_DispathCB = Callable[[], _ReShedule]
# socket conditions
IO_READ = -1
IO_WRITE = -1
IO_EXCEPT = -1


def socket_add(id, method, condition=-1):
	# type: (_FileLike, _SocketCB, int) -> None
	raise NotImplementedError()


def socket_remove(id, condition=-1):
	# type: (_FileLike, int) -> None
	raise NotImplementedError()


def timer_add(interval, method):
	# type: (int, _TimerCB) -> _TimerID
	raise NotImplementedError()


def timer_remove(id):
	# type: (_TimerID) -> None
	raise NotImplementedError()


def dispatcher_add(method, min_timeout=True):
	# type: (_DispathCB, bool) -> Optional[int]
	raise NotImplementedError()


def dispatcher_remove(method):
	# type: (_DispathCB) -> Optional[int]
	raise NotImplementedError()


def loop():
	# type: () -> Any
	raise NotImplementedError()


def step():
	# type: () -> None
	raise NotImplementedError()


# notifier types
(GENERIC, QT, GTK, TWISTED, TORNADO) = range(5)


def init(model=GENERIC, **kwargs):
	# type: (int, **Any) -> None
	global timer_add
	global socket_add
	global dispatcher_add
	global timer_remove
	global socket_remove
	global dispatcher_remove
	global loop, step
	global IO_READ, IO_WRITE, IO_EXCEPT

	if model == GENERIC:
		from . import nf_generic as nf_impl
	elif model == QT:
		from . import nf_qt as nf_impl  # type: ignore
	elif model == GTK:
		from . import nf_gtk as nf_impl  # type: ignore
	elif model == TWISTED:
		from . import nf_twisted as nf_impl  # type: ignore
	elif model == TORNADO:
		from . import nf_tornado as nf_impl  # type: ignore
	else:
		raise ValueError('unknown notifier model')

	socket_add = nf_impl.socket_add
	socket_remove = nf_impl.socket_remove
	timer_add = nf_impl.timer_add
	timer_remove = nf_impl.timer_remove
	dispatcher_add = nf_impl.dispatcher_add
	dispatcher_remove = nf_impl.dispatcher_remove
	loop = nf_impl.loop
	step = nf_impl.step
	IO_READ = nf_impl.IO_READ
	IO_WRITE = nf_impl.IO_WRITE
	IO_EXCEPT = nf_impl.IO_EXCEPT

	if hasattr(nf_impl, '_options') and type(nf_impl._options) == dict:
		for k, v in kwargs.items():
			if k in nf_impl._options:
				nf_impl._options[k] = v

	if hasattr(nf_impl, '_init'):
		nf_impl._init()


class Callback(Generic[_T]):

	def __init__(self, function, *args, **kwargs):
		# type: (Callable[..., _T], *object, **object) -> None
		self._function = function
		self._args = args
		self._kwargs = kwargs

	def __call__(self, *args):
		# type: (*object) -> _T
		tmp = list(args)
		if self._args:
			tmp.extend(self._args)
		return self._function(*tmp, **self._kwargs)

	def __lt__(self, other):
		# type: (Any) -> bool
		return self._function < (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __le__(self, other):
		# type: (Any) -> bool
		return self._function <= (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __eq__(self, other):
		# type: (Any) -> bool
		return self._function == (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __ne__(self, other):
		# type: (Any) -> bool
		return self._function != (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __ge__(self, other):
		# type: (Any) -> bool
		return self._function >= (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __gt__(self, other):
		# type: (Any) -> bool
		return self._function > (other._function if isinstance(other, Callback) else other) if callable(other) else NotImplemented

	def __bool__(self):
		# type: () -> bool
		return bool(self._function)
	__nonzero__ = __bool__

	def __hash__(self):
		# type: () -> int
		return hash(self._function)
